/* -*-c++-*- OpenSceneGraph - Copyright (C) 1998-2006 Robert Osfield
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
*/

#include <osgDB/FileNameUtils>
#include <osgDB/FileUtils>
#include <osgDB/Registry>
#include <osg/Notify>

#include "FreeTypeLibrary.h"

/**
Meta extension that loads and returns freetype face (FT_Face) object (not a file) so that Application can reuse the same FT_Library object
Usage: readObject(".ftfacenew", options);
Note: filename is not taken here but rather from options

Options : (reference https://www.freetype.org/freetype2/docs/reference/ft2-base_interface.html#FT_New_Face)
PluginData        [Input]  -> "index" (unsigned int)
PluginData        [Output] -> "ftface" (FT_Face)
PluginStringData  [Input]  -> "fontPath" (ttf file full path)

Return: returns ReadResult::FILE_LOADED if succesfully loaded
*/
const std::string FT_FACE_NEW_EXT = "ftfacenew";

/**
Meta extension that destroys a given freetype face (FT_Face) object (not a file), destroys the FT_Face created by this FT_Library.
Usage: readObject(".ftfacedone", options);
Note: filename Is not required

Options : (reference https://www.freetype.org/freetype2/docs/reference/ft2-base_interface.html#FT_Done_Face)
PluginData        [Input/Output] -> "ftface" (FT_Face)

Return: returns ReadResult::FILE_LOADED if succesfully loaded, sets NULL to the FT_Face on success
*/
const std::string FT_FACE_DONE_EXT = "ftfacedone";

class ReaderWriterFreeTypeFace : public osgDB::ReaderWriter
{
    public:
        ReaderWriterFreeTypeFace()
        {
            supportsExtension(FT_FACE_NEW_EXT, "create and return ft_face from existing ft_library");
            supportsExtension(FT_FACE_DONE_EXT, "frees ft_face");
        }

        virtual const char* className() const { return "ReaderWriterFreeTypeFace"; }

        virtual ReadResult readObject(const std::string& file, const osgDB::ReaderWriter::Options* options) const
        {
            std::string ext = osgDB::getLowerCaseFileExtension(file);
            if (!acceptsExtension(ext)) return ReadResult::FILE_NOT_HANDLED;

            FreeTypeLibrary* freeTypeLibrary = FreeTypeLibrary::instance();
            if (!freeTypeLibrary)
            {
                OSG_WARN<<"Warning:: cannot create freetype font after freetype library has been deleted."<<std::endl;
                return ReadResult::ERROR_IN_READING_FILE;
            }

            if (ext == FT_FACE_NEW_EXT)
            {
               if (options->getPluginData("index") != NULL && options->getPluginData("ftface") != NULL)
               {
                  const std::string fontPath = options->getPluginStringData("fontPath");
                  const unsigned int& index = *static_cast <const unsigned int*>(options->getPluginData("index"));
                  FT_Face& facePtr = *static_cast <FT_Face*>(const_cast<void*> (options->getPluginData("ftface")));

                  bool success = freeTypeLibrary->getFace(fontPath, index, facePtr);
                  if (success)
                  {
                     return ReadResult::FILE_LOADED;
                  }
                  else
                  {
                     facePtr = NULL;
                  }
               }
               else
               {
                  OSG_WARN << "Warning: Freetype Plugin Data 'index' or 'ftface' were not provided";
               }
               return ReadResult::ERROR_IN_READING_FILE;
            }
            else 
            {
               if (ext == FT_FACE_DONE_EXT)
               {
                  if (options->getPluginData("ftface") != NULL)
                  {
                     FT_Face& facePtr = *static_cast <FT_Face*>(const_cast<void*> (options->getPluginData("ftface")));

                     bool success = freeTypeLibrary->doneFace(facePtr);
                     if (success)
                     {
                        facePtr = NULL;
                        return ReadResult::FILE_LOADED;
                     }
                  }
                  else
                  {
                     OSG_WARN << "Warning: Freetype Plugin Data 'ftface' was not provided";
                  }
                  return ReadResult::ERROR_IN_READING_FILE;
               }
            }
            return ReadResult::ERROR_IN_READING_FILE;
        }
};

// now register with Registry to instantiate the above
// reader/writer.
REGISTER_OSGPLUGIN(freetype_ftface, ReaderWriterFreeTypeFace)
